;; -------------------------------------------------------------------
;; Snowclimber
;; -------------------------------------------------------------------
;; This file contains the actual game engine. Its entry point is the
;; 'game_main' method, which will return after game over. The gameplay
;; itself is managed in the vblank interrupt handler 'game_ontimer'.
;; -------------------------------------------------------------------

work_area:      equ $c900
htimi_save:     equ work_area + $0000
ai_flag:	equ work_area + $0005
ai_mode:	equ work_area + $0006
ai_delay_walk:	equ work_area + $0007
ai_delay_duck:	equ work_area + $0008

player1_duck:   equ work_area + $0009
player2_duck:   equ work_area + $000a
player1_pos:    equ work_area + $000b
player2_pos:    equ work_area + $000d
wind_speed:     equ work_area + $000f
wind_time:	equ work_area + $0010
snow_data:	equ work_area + $0011
	
	;; =================================================================== 
	;; main loop
	;; -------------------------------------------------------------------
	;; This method does the overal management of the game. It initializes
	;; and draws the screen. Enable the interrupt handler that manages the
	;; gameplay and polls the position of the players to see if there is a
	;; winner. If there is a winner it will disable the interrupt handler,
	;; and will display who has won the game. 
	;; =================================================================== 

game_main:	

	;;-------------------------------
	;; build screen
	;;-------------------------------

	call init32
	
	ld   a,[rg1sav]
	or   2
	ld   b,a
	ld   c,1 
	call wrtwdp
	
	call bold_font
	ld   de,3+1*256
	ld   hl,game_text1
	ld   [csry],de
	call prstr
	ld   de,3+25*256
	ld   hl,game_text2
	ld   a,[ai_flag]
	or   a
	jr   z,_game_no_ai
	ld   hl,game_text3
_game_no_ai:
	ld   [csry],de
	call prstr
	ld   hl,game_line
	call prstr
	
	;;-------------------------------
	;; load game tiles
	;;-------------------------------

	ld   hl,game_tiles
	ld   de,128*8
	ld   bc,48
	call ldirmv
	
	;;-------------------------------
	;; display game area
	;;-------------------------------
	
	ld   hl,game_area
	ld   de,$1800 + 8*32
	ld   bc,13*32
	call ldirmv

	;;-------------------------------
	;; load sprites
	;;-------------------------------
	
	ld   hl,game_sprites
	ld   de,$3800
	ld   bc,256
	call ldirmv

	;;-------------------------------
	;; initialize
	;;-------------------------------
	
	ld   de,player1_duck+1
	ld   hl,player1_duck
	ld   bc,110
	xor  a
	ld   [hl],a
	ldir

	call draw_players
    
	;; init ai
	ld   a,1
	ld   [ai_mode],a
;	ld   [ai_flag],a	; force ai
	call init_ai_duck

	;;-------------------------------
	;; countdown
	;;-------------------------------
	
	ld   hl,countdown_text
	ld   b,5
countdown:
	ld   a,50
	call delay
	push bc
	ld   de,22+15*256
	ld   [csry],de
	call prstr
	pop  bc
	djnz countdown

	;;-------------------------------
	;; start game
	;;-------------------------------

	push hl
	call set_interrupt
	ld   a,100
	call delay
	pop  hl
	ld   de,22+15*256
	ld   [csry],de
	call prstr

	;;-------------------------------
	;; check for winner
	;;-------------------------------
	
wait_winner:
	ld   c,0
	ld   de,96 * 8
	ld   hl,[player1_pos]
	rst  $20
	jr   nz,_winner1
	ld   c,1
_winner1:
	ld   hl,[player2_pos]
	rst  $20
	ld   a,c
	jr   nz,_winner2
	or   2
_winner2:
	or   a
	jr   z,wait_winner

	;;-------------------------------
	;; end game
	;;-------------------------------
	
	call restore_interrupt

	;; tell who won
	ld   hl,winner_text1
	dec  a
	jr   z,show_winner
	ld   hl,winner_text3
	dec  a
	jr   nz,show_winner
	ld   hl,winner_text2
	ld   a,[ai_flag]
	or   a
	jr   z,show_winner
	ld   hl,winner_text4
show_winner:
	ld   de,22+10*256
	ld   [csry],de
	call prstr

	;;-------------------------------
	;; wait a few seconds
	;;-------------------------------

	ld   a,150
	call delay
	ret

	;; =================================================================== 
	;; game logic
	;; -------------------------------------------------------------------
	;; This is the method that handles the game logic. It is called every
	;; vblank to make sure the gamespeed is constant.
	;; =================================================================== 

game_ontimer:

	;; -------------------------------
	;; handle snow
	;; -------------------------------
	
	call wind
	call move_snow

	;; -------------------------------
	;; handle player interaction
	;; -------------------------------
	
	call trig_player1
	call trig_player2
	call stick_player1
	call stick_player2
	call move_down

	;; -------------------------------
	;; draw player 1
	;; -------------------------------

draw_players:

	ld   hl,[player1_pos]
	xor  a
	call player_pos2xy
	ld   a,[player1_duck]
	or   a
	ld   a,2
	jr   nz,_duck1
	ld   a,l
	and  1
_duck1:	call put_player1

	;; -------------------------------
	;; draw player 2
	;; -------------------------------
	
	ld   hl,[player2_pos]
	ld   a,1
	call player_pos2xy
	ld   a,[player2_duck]
	or   a
	ld   a,2
	jr   nz,_duck2
	ld   a,l
	and  1
	
_duck2:	
	call put_player2
	ret

	;; -------------------------------------------------------------------
	;; wind
	;; -------------------------------------------------------------------

wind:
	ld   a,[wind_time]
	or   a
	jr   z,_wind1
	dec  a
	ld   [wind_time],a
	or   a
	ret  nz
	;; -------------------------------
	;; wind dropped
	;; -------------------------------
	ld   [wind_speed],a
	ret
_wind1:	
	ld   a,r
	ld   c,a
	call rand8
	cp   c
	ret  nz
	;; -------------------------------
	;; create new wind
	;; -------------------------------
	ld   a,128
	ld   [wind_time],a
	call rand8
	and  7
	ld   c,a
	call rand8
	and  2
	jr   z,_wind2
	;; -------------------------------
	;; right
	;; -------------------------------
	ld   a,c
	ld   [wind_speed],a
	ret
_wind2:
	;; -------------------------------
	;; left
	;; -------------------------------
 	xor  a
 	sub  c
	ld   [wind_speed],a
	ret

	;; -------------------------------------------------------------------
	;; move_snow
	;; -------------------------------------------------------------------

move_snow:
	ld   de,snow_data
	ld   b,25
_move_snow:
	push bc
	;;-------------------------------
	;; y-coordinate
	;;-------------------------------
	ld   a,[de]
	or   a
	call z,create_flake
	cp   160
	call nc,create_flake
	;; -------------------------------
	;; add speed
	;; -------------------------------
	push de
	pop  hl
	inc  hl
	add  a,[hl]
	ld   [de],a
	ld   h,a
	;; -------------------------------
	;; x-coordinate
	;; -------------------------------
	inc  de
	inc  de
	ld   a,[de]
	push hl
	ld   hl,wind_speed
	add  [hl]
	ld   [de],a
	pop  hl
	ld   l,a
	;; -------------------------------
	;; flake type
	;; -------------------------------
	inc  de
	ld   a,[de]
	pop  bc
	ld   c,a
	ld   a,b
	dec  a
	push bc
	ld   b,c
	push de
	;; -------------------------------
	;; display flake
	;; -------------------------------
	call put_snowflake
	pop  de
	pop  bc
	inc  de
	djnz _move_snow
	ret

	;; -------------------------------------------------------------------
	;; create_flake
	;; -------------------------------------------------------------------

create_flake:
	push de
	;; data structure
	;; de + 0 = y
	;; de + 1 = speed
	;; de + 2 = x
	;; de + 3 = flake type
	inc  de
	;; -------------------------------
	;; speed
	;; -------------------------------
	call rand8
	and  3
	inc  a
	ld   [de],a
	inc  de
	;; -------------------------------
	;; x coordinate
	;; -------------------------------
	call rand8
	ld   [de],a
	inc  de
	;; -------------------------------
	;; flake type
	;; -------------------------------
	call rand8
	and  1
	ld   [de],a
	inc  de
	;; -------------------------------
	;; y coordinate
	;; -------------------------------
	pop  de
	ld   a,24
	ld   [de],a
	ret

	;; -------------------------------------------------------------------
	;; stick_player{1,2}
	;; -------------------------------------------------------------------

stick_player1:
	ld   a,[wind_speed]
	or   a
	ret  nz
	ld   a,[player1_duck]
	or   a
	ret  nz			; a=0
	call gtstck
	or   a
	jr   nz,_stick_player1
	inc  a			; a=1
	call gtstck
	or   a
	ret  z
_stick_player1:
	ld   hl,[player1_pos]
	inc  hl
	ld   [player1_pos],hl
	ret

stick_player2:
	ld   a,[wind_speed]
	or   a
	ret  nz
	ld   a,[ai_flag]
	or   a
	jr   nz,stick_player2_ai
	ld   a,[player2_duck]
	or   a
	ret  nz
	ld   a,2
	call gtstck
	or   a
	ret  z
_stick_player2:
	ld   hl,[player2_pos]
	inc  hl
	ld   [player2_pos],hl
	ret

	;; ai
stick_player2_ai:
	ld   a,[ai_mode]
	or   a			; a=0 wait walk
	jr   z,_stick_player2_ai0
	dec  a			; a=1 walk
	jr   z,_stick_player2
	ret
_stick_player2_ai0:
	ld   a,[ai_delay_walk]
	dec  a
	ld   [ai_delay_walk],a		
	ret  nz
	ld   [player2_duck],a
	inc  a			; a=1 walk
	ld   [ai_mode],a
init_ai_duck:	
	call rand8
	and  31
	inc  a
	ld   [ai_delay_duck],a
	ret	
	
	;; -------------------------------------------------------------------
	;; trig_player{1,2}
	;; -------------------------------------------------------------------

trig_player1:
	xor  a			; spacebar
	call gttrig
	or   a
	jr   nz,_trig_player1
	inc  a			; a=1, fire button 1 controler 1
	call gttrig
_trig_player1:
	ld   [player1_duck],a
	ret

trig_player2:
	ld   a,[ai_flag]
	or   a
	jr   nz,trig_player2_ai
	;; regular user handling
	ld   a,2
	call gttrig
	ld   [player2_duck],a
	ret

	;; ai
trig_player2_ai:
	ld   a,[ai_mode]	; a=0 wait walk
	dec  a			; a=1 walk
	jr   z,_trig_player2_ai0
	dec  a			; a=2 wait duck
	jr   z,_trig_player2_ai1
	dec  a			; a=3 duck
	ret  nz			; a=0 wait walk
	inc  a			; a=1 flag duck
	ld   [player2_duck],a
	ld   a,[wind_speed]
	or   a
	ret  nz
	ld   [ai_mode],a	; 0=wait walk
	ret
_trig_player2_ai0:
	ld   a,[wind_speed]
	or   a
	ret  z
	ld   a,2 ; a=2 wait duck
	ld   [ai_mode],a
	ret
_trig_player2_ai1:
	ld   a,[ai_delay_duck]
	dec  a
	ld   [ai_delay_duck],a
	ret  nz
	ld   a,3;   a=3 duck
	ld   [ai_mode],a
	call rand8
	and  15
	inc  a
	ld   [ai_delay_walk],a
	ret

	;; -------------------------------------------------------------------
	;; move_down
	;; -------------------------------------------------------------------

move_down:
	ld   a,[wind_speed]
	or   a
	ret  z
	ld   a,[player1_duck]
	or   a
	jr   nz,_move_down1
	ld   hl,[player1_pos]
	dec  hl
	dec  hl
	ld   a,h
	and  128
	jr   nz,_move_down1
	ld   [player1_pos],hl
_move_down1:
	ld   a,[player2_duck]
	or   a
	ret  nz
	ld   hl,[player2_pos]
	dec  hl
	dec  hl
	ld   a,h
	and  128
	ret  nz
	ld   [player2_pos],hl
	ret

	;; ===================================================================
	;; interrupt helper methods
	;; -------------------------------------------------------------------
	;; The methods below here are some helper methods for the interrupt
	;; handling. It initializes, and restores the interrupt handler to
	;; 'game_ontimer' or its original value.
	;; ===================================================================

	;; -------------------------------------------------------------------
	;; set_interrupt
	;; -------------------------------------------------------------------

set_interrupt:
	di
	ld   hl,htimi
	ld   de,htimi_save
	ld   bc,5
	ldir
 	ld   hl,htimi_new
 	ld   de,htimi
 	ld   bc,5
 	ldir
 	ei
 	ret
	
htimi_new:
 	call game_ontimer
	ret

	;; -------------------------------------------------------------------
	;; restore_interrupt
	;; -------------------------------------------------------------------

restore_interrupt:
	di
	ld   hl,htimi_save
	ld   de,htimi
	ld   bc,5
	ldir
	ei
	ret

	;; ===================================================================
	;; player helper methods
	;; -------------------------------------------------------------------
	;; The methods below here are some helper methods for the main
	;; characters. The position is stored as a indication how far it is
	;; on the track. The 'player_pos2xy' method converts that value to
	;; a x y value to display the player on the screen. The 'put_player'
	;; method displays the actual sprites of the player. The sprites are
	;; 4 8x8 sprites for each player, which is uncommon, and usually not
	;; desired. In this case it saved some bytes to make it fit in as a
	;; 2K game.
	;; ===================================================================

	;; -------------------------------------------------------------------
	;; player_pos2xy
	;; -------------------------------------------------------------------
	;; a  = player (0 or 1)
	;; hl = logical position
	;; -------------------------------------------------------------------

player_pos2xy:
	push af
	;; divide by 8
	srl  h
	rr   l
	srl  h
	rr   l
	srl  h
	rr   l
	;; -------------------------------
	;; y coordinate
	;; -------------------------------
	ld   a,143
	sub  l
	ld   h,a
	pop  af
	or   a
	jr   nz,_player2_pos2xy
	;; -------------------------------
	;; x coordinate
	;; -------------------------------
_player1_pos2xy:
	ld   a,11
	add  a,l
	ld   l,a
	ret
_player2_pos2xy:
	ld   a,229
	sub  l
	ld   l,a
	ret

	;; -------------------------------------------------------------------
	;; put_player{1,2}
	;; -------------------------------------------------------------------
	;;  a = logical sprite no (0-2)
	;;  l = x
	;;  h = y
	;; -------------------------------------------------------------------

put_player1:
	ex   de,hl
	ld   hl,$1b00
	jr   _put_player

put_player2:
	add  a,3
	ex   de,hl
	ld   hl,$1b10

_put_player:
	add  a,a
	add  a,a
	ld   c,a
	;; -------------------------------
	;; sprite 0
	;; -------------------------------
	ld   a,d
	call wrtvrm
	inc  hl
	ld   a,e
	call wrtvrm
	inc  hl
	ld   a,c
	call wrtvrm
	inc  hl
	ld   a,15
	call wrtvrm
	ret

	;; =================================================================== 
	;; snow helper methods 
	;; -------------------------------------------------------------------
	;; The method(s) below are used to display the snowflakes on the
	;; screen.
	;; =================================================================== 

	;; -------------------------------------------------------------------
	;; put_snowflake
	;; -------------------------------------------------------------------
	;;  a = snowflake no (0-23)
	;;  b = type flake   (0-4)
	;;  l = x
	;;  h = y
	;; -------------------------------------------------------------------

put_snowflake:
	push hl
	add  a,a
	add  a,a
	ld   l,a
	ld   h,0
	ld   de,$1b20
	add  hl,de
	pop  de
	ld   a,d
	call wrtvrm
	inc  hl
	ld   a,e
	call wrtvrm
	inc  hl
	ld   a,b
	add  a,a
	add  a,a
	add  24
	call wrtvrm
	inc  hl
	ld   a,15
	call wrtvrm
	ret

	;; =================================================================== 
	;; game data
	;; -------------------------------------------------------------------
	;; Everything below here is data that is used for the visualisation of
	;; the game. 
	;; =================================================================== 

game_line:
	db 133,133,133,133,133,133,133,133,133,133,133,133,133,133,133,133
	db 133,133,133,133,133,133,133,133,133,133,133,133,133,133,133,133,0

countdown_text:
	db "WAIT",0
	db "    ",0
	db "WAIT",0
	db "    ",0
	db " GO ",0
	db "    ",0

game_text1:
	db "PLAYER 1",0
game_text2:
	db "PLAYER 2",0
game_text3:
	db "COMPUTER",0

winner_text1:
	db "PLAYER 1 WON!",0
winner_text2:
	db "PLAYER 2 WON!",0
winner_text4:
	db "COMPUTER WON!",0
winner_text3:
	db "IT'S A DRAW!",0

	;; -------------------------------------------------------------------
	;; game field
	;; -------------------------------------------------------------------

game_area:
	db 128,128,128,128,128,128,128,128,128,128,128,128,128,129,130,130
	db 130,130,131,128,128,128,128,128,128,128,128,128,128,128,128,128
	db 128,128,128,128,128,128,128,128,128,128,128,128,129,130,130,130
	db 130,130,130,131,128,128,128,128,128,128,128,128,128,128,128,128
	db 128,128,128,128,128,128,128,128,128,128,128,129,130,130,130,130
	db 130,130,130,130,131,128,128,128,128,128,128,128,128,128,128,128
	db 128,128,128,128,128,128,128,128,128,128,129,130,130,130,130,130
	db 130,130,130,130,130,131,128,128,128,128,128,128,128,128,128,128
	db 128,128,128,128,128,128,128,128,128,129,130,130,130,130,130,130
	db 130,130,130,130,130,130,131,128,128,128,128,128,128,128,128,128
	db 128,128,128,128,128,128,128,128,129,130,130,130,130,130,130,130
	db 130,130,130,130,130,130,130,131,128,128,128,128,128,128,128,128
	db 128,128,128,128,128,128,128,129,130,130,130,130,130,130,130,130
	db 130,130,130,130,130,130,130,130,131,128,128,128,128,128,128,128
	db 128,128,128,128,128,128,129,130,130,130,130,130,130,130,130,130
	db 130,130,130,130,130,130,130,130,130,131,128,128,128,128,128,128
	db 128,128,128,128,128,129,130,130,130,130,130,130,130,130,130,130
	db 130,130,130,130,130,130,130,130,130,130,131,128,128,128,128,128
	db 128,128,128,128,129,130,130,130,130,130,130,130,130,130,130,130
	db 130,130,130,130,130,130,130,130,130,130,130,131,128,128,128,128
	db 128,128,128,129,130,130,130,130,130,130,130,130,130,130,130,130
	db 130,130,130,130,130,130,130,130,130,130,130,130,131,128,128,128
	db 128,128,129,130,130,130,130,130,130,130,130,130,130,130,130,130
	db 130,130,130,130,130,130,130,130,130,130,130,130,130,131,128,128
	db 132,132,132,132,132,132,132,132,132,132,132,132,132,132,132,132
	db 132,132,132,132,132,132,132,132,132,132,132,132,132,132,132,132
	
	;; -------------------------------------------------------------------
	;; tile data
	;; -------------------------------------------------------------------

game_tiles:
	db 0, 0, 0, 0, 0, 0, 0, 0
	db 1, 3, 7, 15, 31, 63, 127, 255
	db 255, 255, 255, 255, 255, 255, 255, 255
	db 128, 192, 224, 240, 248, 252, 254, 255
	db 255, 255, 255, 255, 255, 255, 255, 0
	db 0, 255, 255, 0, 255, 0, 0, 0

	;; -------------------------------------------------------------------
	;; sprite data
	;; -------------------------------------------------------------------

game_sprites:	
	;; walk 1 player 1
	db 0, 0, 0, 0, 0, 12, 18, 36
	db 35, 68, 130, 146, 162, 132, 120, 32
	db 0, 0, 0, 0, 0, 0, 0, 0
	db 0, 0, 0, 0, 0, 0, 0, 0
	;; walk 2 player 1
	db 0, 0, 0, 0, 0, 12, 18, 36
	db 35, 68, 130, 146, 146, 132, 120, 16
	db 0, 0, 0, 0, 0, 0, 0, 0
	db 0, 0, 0, 0, 0, 0, 0, 0
	;; duck player 1
	db 0, 0, 0, 0, 0, 0, 0, 1
	db 1, 14, 16, 37, 42, 36, 24, 16
	db 0, 0, 0, 0, 0, 0, 192, 32
	db 96, 64, 128, 0, 0, 0, 0, 0
	;; walk 1 player 2
	db 0, 0, 0, 0, 0, 0, 0, 0
	db 0, 0, 0, 0, 0, 0, 0, 0
	db 0, 0, 0, 0, 0, 48, 72, 36
	db 196, 34, 65, 73, 69, 33, 30, 4
	;; walk 2 player 2
	db 0, 0, 0, 0, 0, 0, 0, 0
	db 0, 0, 0, 0, 0, 0, 0, 0
	db 0, 0, 0, 0, 0, 48, 72, 36
	db 196, 34, 65, 73, 73, 33, 30, 8
	;; duck player 2
	db 0, 0, 0, 0, 0, 0, 3, 4
	db 6, 2, 1, 0, 0, 0, 0, 0
	db 0, 0, 0, 0, 0, 0, 0, 128
	db 128, 112, 8, 164, 84, 36, 24, 8
	;; snow flakes
	db 128, 0, 0, 0, 0, 0, 0, 0
	db 0, 0, 0, 0, 0, 0, 0, 0
	db 0, 0, 0, 0, 0, 0, 0, 0
	db 0, 0, 0, 0, 0, 0, 0, 0
	db 192, 192, 0, 0, 0, 0, 0, 0
	db 0, 0, 0, 0, 0, 0, 0, 0
	db 0, 0, 0, 0, 0, 0, 0, 0
	db 0, 0, 0, 0, 0, 0, 0, 0


